from __future__ import absolute_import

import math

import torch
import torch.nn as nn
import torch.nn.init as init
import torch.nn.functional as F
import copy

__all__ = [
    "resnet32_2head",
    "resnet56_2head",
    "resnet32_4head",
    "resnet56_4head",
    "resnet18_2head",
    "resnet50_2head",
    "resnet101_2head",
    "resnet152_2head",
    "resnet18_4head",
    "resnet50_4head",
    "resnet101_4head",
    "resnet152_4head",
    "resnet18_8head",
    "resnet50_8head",
    "resnet101_8head",
    "resnet152_8head",
    "resnet32",
    "resnet56",
    "resnet18",
    "resnet50",
    "resnet101",
    "resnet152",
    'MobileNetV2',
    'mobilenetv2_19',
]


def conv3x3(in_planes, out_planes, stride=1):
    "3x3 convolution with padding"
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=1, bias=False)

def conv1x1(in_planes, planes, stride=1):
    return nn.Conv2d(in_planes, planes, kernel_size=1, stride=stride, bias=False)

class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out

class Bottleneck(nn.Module):

    def __init__(self, inplanes, planes, stride=1, downsample=None, expansion=1):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(inplanes, inplanes*expansion, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(inplanes*expansion)
        self.conv2 = nn.Conv2d(inplanes*expansion, inplanes*expansion, kernel_size=3, stride=stride,
                               padding=1, bias=False, groups=inplanes*expansion)
        self.bn2 = nn.BatchNorm2d(inplanes*expansion)
        self.conv3 = nn.Conv2d(inplanes*expansion, planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(planes)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out

class BottleneckBlock(nn.Module):
    expansion = 4
    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(BottleneckBlock, self).__init__()
        self.conv1 = conv1x1(inplanes, planes)
        self.bn1 = nn.BatchNorm2d(planes)
        self.relu = nn.ReLU(inplace=True)

        self.conv2 = conv3x3(planes, planes, stride)
        self.bn2 = nn.BatchNorm2d(planes)

        self.conv3 = conv1x1(planes, planes*self.expansion)
        self.bn3 = nn.BatchNorm2d(planes*self.expansion)

        self.downsample = downsample
        self.stride = stride
    
    def forward(self, x):
        residual = x

        output = self.conv1(x)
        output = self.bn1(output)
        output = self.relu(output)

        output = self.conv2(output)
        output = self.bn2(output)
        output = self.relu(output)

        output = self.conv3(output)
        output = self.bn3(output)
        
        if self.downsample is not None:
            residual = self.downsample(x)

        output += residual
        output = self.relu(output)

        return output    
    
    

class ResNet_1_4head(nn.Module):
    """Resnet model

    Args:
        block (class): block type, BasicBlock or BottleneckBlock
        layers (int list): layer num in each block
        num_classes (int): class num
    """

    def __init__(self, block, layers, num_classes=1000):
        super(ResNet_1_4head, self).__init__()           ################Symme_ResNet -> ResNet_1_4head
        self.inplanes = 64
        self.conv1 = nn.Conv2d(3, self.inplanes, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(self.inplanes)
        self.relu = nn.ReLU(inplace=True)
        # self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)

        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2_1 = self._make_layer(block, 128, layers[1], stride=2)
        self.layer3_1 = self._make_layer(block, 256, layers[2], stride=2)
        self.layer4_1 = self._make_layer(block, 512, layers[3], stride=2)


        self.layer2_2 = copy.deepcopy(self.layer2_1)
        self.layer3_2 = copy.deepcopy(self.layer3_1)
        self.layer3_3 = copy.deepcopy(self.layer3_1)
        self.layer3_4 = copy.deepcopy(self.layer3_1)
        self.layer4_2 = copy.deepcopy(self.layer4_1)
        self.layer4_3 = copy.deepcopy(self.layer4_1)
        self.layer4_4 = copy.deepcopy(self.layer4_1)


        self.avgpool1 = nn.AdaptiveAvgPool2d((1,1))
        self.middle_fc1 = nn.Linear(512 * block.expansion, num_classes)


        self.avgpool2 = nn.AdaptiveAvgPool2d((1,1))
        self.middle_fc2 = nn.Linear(512 * block.expansion, num_classes)



        self.avgpool3 = nn.AdaptiveAvgPool2d((1,1))
        self.middle_fc3 = nn.Linear(512 * block.expansion, num_classes)

        self.avgpool4 = nn.AdaptiveAvgPool2d((1,1))
        self.middle_fc4 = nn.Linear(512 * block.expansion, num_classes)


        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

    def _make_layer(self, block, planes, layers, stride=1):
        """A block with 'layers' layers

        Args:
            block (class): block type
            planes (int): output channels = planes * expansion
            layers (int): layer num in the block
            stride (int): the first layer stride in the block
        """
        downsample = None
        if stride !=1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                nn.BatchNorm2d(planes * block.expansion),
            )
        layer = []
        layer.append(block(self.inplanes, planes, stride=stride, downsample=downsample))
        self.inplanes = planes * block.expansion
        for i in range(1, layers):
            layer.append(block(self.inplanes, planes))

        return nn.Sequential(*layer)

    def forward(self, x):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        # x = self.maxpool(x)

        x = self.layer1(x)
        middle_output1 = self.layer2_1(x)
        middle_output1 = self.layer3_1(middle_output1)
        middle_output1 = self.layer4_1(middle_output1)
        middle_output1 = self.avgpool1(middle_output1)
        middle_output1 = torch.flatten(middle_output1, 1)
        middle_output1 = self.middle_fc1(middle_output1)
        torch.cuda.empty_cache()
        middle_output2 = self.layer2_1(x)
        middle_output2 = self.layer3_2(middle_output2)
        middle_output2 = self.layer4_2(middle_output2)
        middle_output2 = self.avgpool2(middle_output2)
        middle_output2 = torch.flatten(middle_output2, 1)
        middle_output2 = self.middle_fc2(middle_output2)
        torch.cuda.empty_cache()
        middle_output3 = self.layer2_2(x)
        middle_output3 = self.layer3_3(middle_output3)
        middle_output3 = self.layer4_3(middle_output3)
        middle_output3 = self.avgpool3(middle_output3)
        middle_output3 = torch.flatten(middle_output3, 1)
        middle_output3 = self.middle_fc3(middle_output3)
        torch.cuda.empty_cache()
        middle_output4 = self.layer2_2(x)
        middle_output4 = self.layer3_4(middle_output4)
        middle_output4 = self.layer4_4(middle_output4)
        middle_output4 = self.avgpool4(middle_output4)
        middle_output4 = torch.flatten(middle_output4, 1)
        middle_output4 = self.middle_fc4(middle_output4)
        torch.cuda.empty_cache()

        return middle_output1, middle_output2, middle_output3, middle_output4



class ResNet_1_ori(nn.Module):
    """Resnet model
    #18 50 101
    Args:
        block (class): block type, BasicBlock or BottleneckBlock
        layers (int list): layer num in each block
        num_classes (int): class num
    """

    def __init__(self, block, layers, num_classes=1000):
        super(ResNet_1_ori, self).__init__()
        self.inplanes = 64
        self.conv1 = nn.Conv2d(3, self.inplanes, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(self.inplanes)
        self.relu = nn.ReLU(inplace=True)
        # self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)

        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2_1 = self._make_layer(block, 128, layers[1], stride=2)
        self.layer3_1 = self._make_layer(block, 256, layers[2], stride=2)
        self.layer4_1 = self._make_layer(block, 512, layers[3], stride=2)

        self.avgpool1 = nn.AdaptiveAvgPool2d((1,1))
        self.middle_fc1 = nn.Linear(512 * block.expansion, num_classes)


        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, nn.BatchNorm2d): 
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)
        
    def _make_layer(self, block, planes, layers, stride=1):
        """A block with 'layers' layers

        Args:
            block (class): block type
            planes (int): output channels = planes * expansion
            layers (int): layer num in the block
            stride (int): the first layer stride in the block
        """
        downsample = None
        if stride !=1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                nn.BatchNorm2d(planes * block.expansion),
            )
        layer = []
        layer.append(block(self.inplanes, planes, stride=stride, downsample=downsample))
        self.inplanes = planes * block.expansion
        for i in range(1, layers):
            layer.append(block(self.inplanes, planes))
        
        return nn.Sequential(*layer)
    
    def forward(self, x):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        # x = self.maxpool(x)

        x = self.layer1(x)
        middle_output1 = self.layer2_1(x)
        middle_output1 = self.layer3_1(middle_output1)     
        middle_output1 = self.layer4_1(middle_output1)           
        middle_output1 = self.avgpool1(middle_output1)
        middle_output1 = torch.flatten(middle_output1, 1)
        middle_output1 = self.middle_fc1(middle_output1)        
        

        return middle_output1




def resnet18_4head(num_classes=1000):
    """
    Constructs a ResNet model.
    """
    return ResNet_1_4head(BasicBlock, [2,2,2,2], num_classes=num_classes) #################resnet diff inital 
    # return ResNet_1_4head(depth=18, num_classes=num_classes)

def resnet50_4head(num_classes=1000):
    """
    Constructs a ResNet model.
    """
    return ResNet_1_4head(BottleneckBlock, [3,4,6,3], num_classes=num_classes)################



def resnet18(num_classes=1000):
    """
    Constructs a ResNet model.
    """
    return ResNet_1_ori(BasicBlock, [2,2,2,2], num_classes=num_classes)

def resnet50(num_classes=1000):
    """
    Constructs a ResNet model.
    """
    return ResNet_1_ori(BottleneckBlock, [3,4,6,3], num_classes=num_classes)
